<?php
// ARCHIVO: SuperAdminController.php - VERSIÓN CORREGIDA
// VERSIÓN: UrbanOSS 3.1 - Laravel 11.47
// FECHA: 2024-12-08
// CORRECCIONES: Imports, validaciones, manejo de excepciones

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\View\View;
use Illuminate\Support\Facades\Log;
use App\Models\User;
use App\Models\Condominio;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;

class SuperAdminController extends Controller
{
    /**
     * Dashboard del SuperAdmin
     */
    public function dashboard(): View
    {
        try {
            $stats = [
                'totalCondominios' => Condominio::count(),
                'activeCondominios' => Condominio::where('estado', 'activo')->count(),
                'totalUsers' => User::count(),
                'pendingUsers' => User::where('estado', 'pendiente')->count(),
            ];
            
            return view('admin.superadmin.dashboard', [
                'pageTitle' => 'Dashboard SuperAdmin - UrbanOSS',
                'stats' => $stats
            ]);
        } catch (\Exception $e) {
            Log::error('Error en dashboard SuperAdmin: ' . $e->getMessage());
            return view('admin.superadmin.dashboard', [
                'pageTitle' => 'Dashboard SuperAdmin - UrbanOSS',
                'stats' => [],
                'error' => 'Error al cargar estadísticas'
            ]);
        }
    }
    
    /**
     * Lista de administradores
     */
    public function administrators(): View
    {
        try {
            $administrators = User::where('role', 'admin')
                ->with('condominio')
                ->orderBy('name')
                ->get();
                
            return view('admin.superadmin.administrators', [
                'pageTitle' => 'Administradores - UrbanOSS',
                'administrators' => $administrators
            ]);
        } catch (\Exception $e) {
            Log::error('Error al cargar administradores: ' . $e->getMessage());
            return view('admin.superadmin.administrators', [
                'pageTitle' => 'Administradores - UrbanOSS',
                'administrators' => collect()
            ]);
        }
    }
    
    /**
     * Crear nuevo administrador
     */
    public function storeAdministrator(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'name' => 'required|string|max:255',
                'email' => 'required|email|unique:users,email',
                'condominio_id' => 'required|exists:condominios,id',
                'password' => 'required|min:8|confirmed',
            ]);
            
            if ($validator->fails()) {
                return redirect()->back()
                    ->withErrors($validator)
                    ->withInput();
            }
            
            $user = User::create([
                'name' => $request->name,
                'email' => $request->email,
                'password' => Hash::make($request->password),
                'role' => 'admin',
                'condominio_id' => $request->condominio_id,
                'estado' => 'activo',
                'email_verified_at' => now(),
            ]);
            
            Log::info('Nuevo administrador creado', [
                'id' => $user->id,
                'email' => $user->email,
                'by' => auth()->id()
            ]);
            
            return back()->with('success', 'Administrador creado exitosamente.');
        } catch (\Exception $e) {
            Log::error('Error al crear administrador: ' . $e->getMessage());
            return back()->with('error', 'Error al crear administrador: ' . $e->getMessage());
        }
    }
    
    /**
     * Eliminar administrador
     */
    public function destroyAdministrator($id)
    {
        try {
            $user = User::findOrFail($id);
            
            // No permitir eliminarse a sí mismo
            if ($user->id === auth()->id()) {
                return back()->with('error', 'No puedes eliminarte a ti mismo.');
            }
            
            // Solo eliminar administradores
            if ($user->role !== 'admin') {
                return back()->with('error', 'El usuario no es un administrador.');
            }
            
            $user->delete();
            
            Log::info('Administrador eliminado', [
                'id' => $id,
                'by' => auth()->id()
            ]);
            
            return back()->with('success', 'Administrador eliminado exitosamente.');
        } catch (\Exception $e) {
            Log::error('Error al eliminar administrador: ' . $e->getMessage());
            return back()->with('error', 'Error al eliminar administrador.');
        }
    }
    
    /**
     * Reportes financieros
     */
    public function financialReports(): View
    {
        return view('admin.superadmin.reports.financial', [
            'pageTitle' => 'Reportes Financieros - UrbanOSS'
        ]);
    }
    
    /**
     * Reportes operativos
     */
    public function operationalReports(): View
    {
        return view('admin.superadmin.reports.operational', [
            'pageTitle' => 'Reportes Operativos - UrbanOSS'
        ]);
    }
    
    /**
     * Exportar reportes
     */
    public function exportReports($type)
    {
        try {
            $filePath = storage_path("app/reports/report-$type.xlsx");
            
            if (!file_exists($filePath)) {
                return back()->with('error', 'El reporte solicitado no existe.');
            }
            
            return response()->download($filePath);
        } catch (\Exception $e) {
            Log::error('Error al exportar reporte: ' . $e->getMessage());
            return back()->with('error', 'Error al exportar reporte.');
        }
    }
    
    /**
     * Configuración global
     */
    public function settings(): View
    {
        return view('admin.superadmin.settings', [
            'pageTitle' => 'Configuración Global - UrbanOSS'
        ]);
    }
    
    /**
     * Actualizar configuración
     */
    public function updateSettings(Request $request)
    {
        try {
            // Implementar lógica de configuración
            Log::info('Configuración actualizada', [
                'data' => $request->except(['_token', '_method']),
                'by' => auth()->id()
            ]);
            
            return back()->with('success', 'Configuración actualizada exitosamente.');
        } catch (\Exception $e) {
            Log::error('Error al actualizar configuración: ' . $e->getMessage());
            return back()->with('error', 'Error al actualizar configuración.');
        }
    }
    
    /**
     * Logs de auditoría
     */
    public function auditLogs(): View
    {
        return view('admin.superadmin.audit-logs', [
            'pageTitle' => 'Logs de Auditoría - UrbanOSS'
        ]);
    }
    
    /**
     * Logs de actividad
     */
    public function activityLogs(): View
    {
        return view('admin.superadmin.activity-logs', [
            'pageTitle' => 'Logs de Actividad - UrbanOSS'
        ]);
    }
}