<?php
// ARCHIVO: LoginController.php - VERSIÓN SEGURA
// VERSIÓN: UrbanOSS 3.1 - Laravel 11.47
// FECHA: 2024-12-08
// CORRECCIÓN: Removido selector de condominios por seguridad

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\URL;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Log;
use App\Models\Condominio;
use Illuminate\View\View;

class LoginController extends Controller
{
    /**
     * Mostrar formulario de login (SIN selector de condominios)
     */
    public function showLoginForm(): View
    {
        try {
            // Generar token CSRF
            $csrfToken = csrf_token();
            
            // Información de registro (sin token)
            $registrationInfo = [
                'available' => false,
                'message' => 'Para registrarse, contacte al administrador de su condominio.',
                'contact_email' => 'soporte@urbanoss.com'
            ];
            
            // URL para recuperación de contraseña
            $passwordResetUrl = URL::route('password.request');
            
            return view('auth.login', [
                'pageTitle' => 'Iniciar Sesión - UrbanOSS',
                'csrfToken' => $csrfToken,
                'registrationInfo' => $registrationInfo,
                'passwordResetUrl' => $passwordResetUrl,
                'error' => session('error'),
                'success' => session('success'),
                'info' => session('info'),
                'condominio' => session('condominio'),
            ]);
            
        } catch (\Exception $e) {
            Log::error('Error en showLoginForm: ' . $e->getMessage());
            
            return view('auth.login', [
                'pageTitle' => 'Iniciar Sesión - UrbanOSS',
                'csrfToken' => csrf_token(),
                'registrationInfo' => [
                    'available' => false,
                    'message' => 'Sistema temporalmente no disponible.',
                    'contact_email' => 'soporte@urbanoss.com'
                ],
                'passwordResetUrl' => '#',
                'error' => 'Error al cargar el formulario de login.',
            ]);
        }
    }
    
    /**
     * Procesar login (SIN condominio_id)
     */
    public function login(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'email' => 'required|email',
                'password' => 'required',
            ]);
            
            if ($validator->fails()) {
                return redirect()->back()
                    ->withErrors($validator)
                    ->withInput($request->except('password'));
            }
            
            $credentials = $request->only('email', 'password');
            
            // Intentar autenticación
            if (Auth::attempt($credentials, $request->filled('remember'))) {
                $user = Auth::user();
                
                // Verificar estado del usuario
                if ($user->estado !== 'activo') {
                    Auth::logout();
                    return redirect()->route('login')
                        ->with('error', 'Su cuenta está inactiva. Contacte al administrador.');
                }
                
                // Verificar condominio (excepto superadmin)
                if ($user->role !== 'superadmin') {
                    if (!$user->condominio_id) {
                        Auth::logout();
                        return redirect()->route('login')
                            ->with('error', 'No tiene un condominio asignado.');
                    }
                    
                    $condominio = Condominio::find($user->condominio_id);
                    if (!$condominio || $condominio->estado !== 'activo') {
                        Auth::logout();
                        return redirect()->route('login')
                            ->with('error', 'El condominio asociado está inactivo.');
                    }
                }
                
                // Regenerar sesión
                $request->session()->regenerate();
                
                // Registrar login exitoso
                Log::info('Login exitoso', [
                    'user_id' => $user->id,
                    'email' => $user->email,
                    'role' => $user->role,
                    'ip' => $request->ip(),
                ]);
                
                // Redirigir al dashboard según rol
                return $this->redirectToDashboard($user);
                
            } else {
                // Login fallido
                Log::warning('Intento de login fallido', [
                    'email' => $request->email,
                    'ip' => $request->ip(),
                ]);
                
                return redirect()->route('login')
                    ->with('error', 'Credenciales incorrectas.')
                    ->withInput($request->except('password'));
            }
            
        } catch (\Exception $e) {
            Log::error('Error en proceso de login: ' . $e->getMessage());
            
            return redirect()->route('login')
                ->with('error', 'Error en el sistema. Intente nuevamente.')
                ->withInput($request->except('password'));
        }
    }
    
    /**
     * Redirigir al dashboard según rol
     */
    protected function redirectToDashboard($user)
    {
        switch ($user->role) {
            case 'superadmin':
                return redirect()->route('superadmin.dashboard');
            case 'admin':
                return redirect()->route('admin.dashboard');
            case 'propietario':
            case 'inquilino':
                return redirect()->route('resident.dashboard');
            case 'guardia':
                return redirect()->route('security.dashboard');
            default:
                Auth::logout();
                return redirect()->route('login')
                    ->with('error', 'Rol de usuario no válido.');
        }
    }
    
    /**
     * Logout
     */
    public function logout(Request $request)
    {
        $user = Auth::user();
        
        if ($user) {
            Log::info('Logout', [
                'user_id' => $user->id,
                'email' => $user->email,
            ]);
        }
        
        Auth::logout();
        $request->session()->invalidate();
        $request->session()->regenerateToken();
        
        return redirect()->route('home');
    }
    
    /**
     * Verificar autenticación (para AJAX)
     */
    public function checkAuth()
    {
        $user = Auth::user();
        
        return response()->json([
            'authenticated' => Auth::check(),
            'user' => $user ? [
                'id' => $user->id,
                'name' => $user->name,
                'email' => $user->email,
                'role' => $user->role,
            ] : null,
        ]);
    }
    
    /**
     * Login específico para condominio (para links externos)
     */
    public function condominioLogin($condominioId)
    {
        try {
            $condominio = Condominio::where('id', $condominioId)
                ->where('estado', 'activo')
                ->firstOrFail();
            
            Session::put('condominio_login_id', $condominioId);
            
            return redirect()->route('login')
                ->with('condominio', $condominio)
                ->with('info', 'Ingrese sus credenciales para acceder a ' . $condominio->nombre);
                
        } catch (\Exception $e) {
            Log::error('Error en condominioLogin: ' . $e->getMessage());
            
            return redirect()->route('login')
                ->with('error', 'Condominio no encontrado o inactivo.');
        }
    }
    
    /**
     * Mostrar formulario de registro público (solo con token válido)
     */
    public function showRegistrationForm($token = null)
    {
        if (!$token) {
            return redirect()->route('login')
                ->with('info', 'Para registrarse, necesita un token de invitación válido proporcionado por el administrador.');
        }
        
        // Validar token aquí (implementar según tu lógica)
        
        return view('auth.register', [
            'pageTitle' => 'Registro - UrbanOSS',
            'token' => $token,
        ]);
    }
    
    /**
     * Mostrar formulario de recuperación de contraseña
     */
    public function showLinkRequestForm(): View
    {
        return view('auth.passwords.email', [
            'pageTitle' => 'Recuperar Contraseña - UrbanOSS'
        ]);
    }
    
    /**
     * Mostrar formulario de reset de contraseña
     */
    public function showResetForm($token): View
    {
        return view('auth.passwords.reset', [
            'pageTitle' => 'Restablecer Contraseña - UrbanOSS',
            'token' => $token
        ]);
    }
    
    /**
     * Página de acceso no autorizado
     */
    public function unauthorized(): View
    {
        return view('errors.unauthorized', [
            'pageTitle' => 'Acceso No Autorizado - UrbanOSS'
        ]);
    }
}